#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\LnLst.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\WsLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'Long/WString ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = LnWsTreNew() 'get handle for new container
    '   h = LnWsTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro LnWsTreNodeTag = 217604277
Macro LnWsTreTag = 1868553327
Type LnWsTreNode
    tag As Long
    P As LnWsTreNode Ptr
    L As LnWsTreNode Ptr
    R As LnWsTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Long
End Type
Type LnWsTre
    tag As Long
    count As Long
    root As LnWsTreNode Ptr
End Type

Function LnWsTreNew() As Long
    'allocate new container - return handle
    Local p As LnWsTre Ptr
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = LnWsTreTag
    Function = p
End Function

Function LnWsTreFinal(ByVal pTree As LnWsTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>LnWsTreTag, LibErrH)
        LnWsTreClear pTree
        MemFree(pTree)
    End If
End Function

Function LnWsTreValidate(ByVal pTree As LnWsTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = LnWsTreTag Then Function = @pTree.tag
End Function

Sub LnWsTreClear(ByVal pTree As LnWsTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As LnWsTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = LnWsTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = LnWsTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            LnWsTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function LnWsTreCount(ByVal pTree As LnWsTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub LnWsTreSet(ByVal pTree As LnWsTre Ptr, ByVal key As Long, ByRef value As WString, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local n As LnWsTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    If @pTree.root Then
        n = @pTree.root
        While 1
            If key > @n.K Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = LnWsTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    @n.@R.K = key
                    WsSet @n.@R.V, value
                    LnWsTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf key < @n.K Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = LnWsTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    @n.@L.K = key
                    WsSet @n.@L.V, value
                    LnWsTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then WsSet @n.V, value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = LnWsTreAllocNode(pTree) : If Err Then Exit Sub
        @pTree.@root.K = key
        WsSet @pTree.@root.V, value
        @pTree.count = 1
    End If
End Sub

Function LnWsTreGet(ByVal pTree As LnWsTre Ptr, ByVal key As Long) As WString
    'get Key's associated Value
    Register k As Long : k = key
    Local n As LnWsTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = WsGet(@n.V)
            Exit Loop
        End If
    Wend
End Function

Function LnWsTreGot(ByVal pTree As LnWsTre Ptr, ByVal key As Long) As Long
    'True/False if Key exist
    Register k As Long : k = key
    Local n As LnWsTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    n = @pTree.root
    While n
        If k < @n.K Then
            n = @n.L
        ElseIf k > @n.K Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
End Function

Sub LnWsTreDel(ByVal pTree As LnWsTre Ptr, ByVal key As Long)
    'remove Key and associated Value
    Local pNode As LnWsTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    pNode = LnWsTreGot(pTree, key)
    If pNode Then
        LnWsTreRemoveNode(pTree, pNode)
    End If
End Sub

Function LnWsTreFirst(ByVal pTree As LnWsTre Ptr) As Long
    'get handle to first node in tree
    Local n As LnWsTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function LnWsTreLast(ByVal pTree As LnWsTre Ptr) As Long
    'get handle to last node in tree
    Local n As LnWsTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function LnWsTreNext(ByVal pNode As LnWsTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As LnWsTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnWsTreNodeTag, LibErrH)
        minR = LnWsTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = LnWsTreParentGreater(pNode)
        End If
    End If
End Function

Function LnWsTrePrev(ByVal pNode As LnWsTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As LnWsTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>LnWsTreNodeTag, LibErrH)
        maxL = LnWsTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = LnWsTreParentLesser(pNode)
        End If
    End If
End Function

Function LnWsTreGetKey(ByVal pNode As LnWsTreNode Ptr) As Long
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>LnWsTreNodeTag, LibErrH)
    Function = @pNode.K
End Function

Function LnWsTreGetVal(ByVal pNode As LnWsTreNode Ptr) As WString
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>LnWsTreNodeTag, LibErrH)
    Function = WsGet(@pNode.V)
End Function

Sub LnWsTreSetVal(ByVal pNode As LnWsTreNode Ptr, ByRef value As WString)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>LnWsTreNodeTag, LibErrH)
    WsSet @pNode.V, value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function LnWsTreClone(ByVal pTree As LnWsTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    clone = LnWsTreNew() : If Err Then Exit Function
    h = LnWsTreFirst(pTree)
    While h
        LnWsTreSet clone, LnWsTreGetKey(h), LnWsTreGetVal(h)
        h = LnWsTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function LnWsTreStore(ByVal pTree As LnWsTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    keys = LnLstNew() : If Err Then Exit Function
    vals = WsLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = LnWsTreFirst(pTree)
        While h
            LnLstAdd keys, LnWsTreGetKey(h)
            WsLstAdd vals, LnWsTreGetVal(h)
            h = LnWsTreNext(h)
        Wend
        SsLstAdd stor, LnLstStore(keys)
        SsLstAdd stor, WsLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = LnLstFinal(keys)
    vals = WsLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub LnWsTreRestore(ByVal pTree As LnWsTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    LnWsTreClear pTree
    keys = LnLstNew() : If Err Then Exit Sub
    vals = WsLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        LnLstRestore keys, SsLstPopFirst(stor)
        WsLstRestore vals, SsLstPopFirst(stor)
        ExitS(LnLstCount(keys)<>WsLstCount(vals), LibErrU)
        While LnLstCount(keys)
            LnWsTreSet pTree, LnLstPopFirst(keys), WsLstPopFirst(vals)
        Wend
    End If
    keys = LnLstFinal(keys)
    vals = WsLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub LnWsTreFileStore(ByVal pTree As LnWsTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    s = LnWsTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub LnWsTreFileRestore(ByVal pTree As LnWsTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>LnWsTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        LnWsTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub LnWsTreRemoveNode(ByVal p As LnWsTre Ptr, ByVal n As LnWsTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As LnWsTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, LnWsTreMaxLeft(n), LnWsTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        LnWsTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = LnWsTreFreeNode(p, n)
        LnWsTreBalanceBranch p, nP
    End If
End Sub

Function LnWsTreAllocNode(ByVal p As LnWsTre Ptr) Private As Long
    Local n As LnWsTreNode Ptr
    n = MemAlloc(SizeOf(LnWsTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.V = WsNew() : If Err Then Exit Function
    @n.tag = LnWsTreNodeTag
    Incr @p.count
    Function = n
End Function

Function LnWsTreFreeNode(ByVal p As LnWsTre Ptr, ByVal n As LnWsTreNode Ptr) As Long
    If n Then
        @n.V = WsFinal(@n.V)
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub LnWsTreBalanceBranch(ByVal p As LnWsTre Ptr, ByVal n As LnWsTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = LnWsTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = LnWsTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function LnWsTreMaxLeft(ByVal n As LnWsTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function LnWsTreMinRight(ByVal n As LnWsTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function LnWsTreParentGreater(ByVal n As LnWsTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnWsTreParentLesser(ByVal n As LnWsTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function LnWsTreRotateLeft(ByVal p As LnWsTre Ptr, ByVal n As LnWsTreNode Ptr) Private As Long
    Local nR, nRL As LnWsTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function LnWsTreRotateRight(ByVal p As LnWsTre Ptr, ByVal n As LnWsTreNode Ptr) Private As Long
    Local nL, nLR As LnWsTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
