#Include This Once
#Include Once "C:\HLib3\HLib.inc"
#Include Once "C:\HLib3\List\SsLst.inc"
#Include Once "C:\HLib3\List\DbLst.inc"

'++
    '----------------------------------------------------------------------------------------
    'String/Double ~ Tree Container
    '   - also called: Tree/Map/Dictionary/Associative Array
    '   - AVL Self-Balanced Binary Tree
    '   - one-to-one relationship
    '   - Key/Value data structure
    '   - Values stored/retrieved/removed using unique lookup Key
    '   - Keys must be unique
    '   - no limit on Key length
    '   - use SsDbTreComparison() to change how Keys compared
    '   - Value replaced if Key exist unless DontReplace = True
    '   - Tree always stays in Key order
    '   - Tree may be traversed forward/backward in Key order
    '   - Tree is self-balanced to maintain shortest average path to each Key
    '
    '   - use MultiTree for one-to-many relationship
    '
    '   container accessed with handle
    '   handle protected by hash tag
    '   h = SsDbTreNew() 'get handle for new container
    '   h = SsDbTreFinal(h) 'free handle before it goes out of scope
    '----------------------------------------------------------------------------------------
'--


Macro SsDbTreNodeTag = -439488143
Macro SsDbTreTag = -252380731
Type SsDbTreNode
    tag As Long
    P As SsDbTreNode Ptr
    L As SsDbTreNode Ptr
    R As SsDbTreNode Ptr
    HL As Word
    HR As Word
    K As Long
    V As Double
End Type
Type SsDbTre
    tag As Long
    count As Long
    root As SsDbTreNode Ptr
    compareCB As Long
    collation As SsStr Ptr
End Type

Function SsDbTreNew() As Long
    'allocate new container - return handle
    Local p As SsDbTre Ptr
    Err = 0
    p = MemAlloc(SizeOf(@p))
    ExitF(p=0, LibErrM)
    @p.tag = SsDbTreTag
    @p.compareCB = CodePtr(SsCompare)
    @p.collation = SsNew() : If Err Then Exit Function
    Function = p
End Function

Function SsDbTreFinal(ByVal pTree As SsDbTre Ptr) As Long
    'free allocated container - return null
    If pTree Then
        ExitF(@pTree.tag<>SsDbTreTag, LibErrH)
        @pTree.collation = SsFinal(@pTree.collation)
        SsDbTreClear pTree
        MemFree(pTree)
    End If
End Function

Function SsDbTreValidate(ByVal pTree As SsDbTre Ptr) As Long
    'True/False if valid handle for this container
    If pTree And @pTree.tag = SsDbTreTag Then Function = @pTree.tag
End Function

Sub SsDbTreComparison(ByVal pTree As SsDbTre Ptr, ByVal compareUCase As Long, ByVal collationSequence As String)
    'set how Strings compared
    'default = case ignored
    'if collationSequence String provided then
    '   Strings are compared using the order of the collation sequence String
    '   collation String must be 256 characters
    'else if compareUCase = True then
    '   Strings compared UCase
    ExitS(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    @pTree.compareCB = CodePtr(SsCompare)
    SsClear @pTree.collation
    If Len(collationSequence) Then
        ExitS(Len(collationSequence)<>256, LibErrS)
        SsSet @pTree.collation, collationSequence : If Err Then Exit Sub
        @pTree.compareCB = CodePtr(SsCompareCollate)
    ElseIf compareUCase Then
        @pTree.compareCB = CodePtr(SsCompareUCase)
    End If
End Sub

Sub SsDbTreClear(ByVal pTree As SsDbTre Ptr)
    'delete all data
    Local i As Long
    Local pNode As SsDbTreNode Ptr
    Local nodes() As Long
    ExitS(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    If @pTree.count Then
        ReDim nodes(1 To @pTree.count)
        i = 0
        pNode = SsDbTreFirst(pTree)
        While pNode
            Incr i
            nodes(i) = pNode
            pNode = SsDbTreNext(pNode)
        Wend
        For i = 1 To @pTree.count
            SsDbTreFreeNode(pTree, nodes(i))
        Next i
    End If
    @pTree.count = 0
    @pTree.root = 0
End Sub

Function SsDbTreCount(ByVal pTree As SsDbTre Ptr) As Long
    'get item count
    If pTree Then Function = @pTree.count
End Function

Sub SsDbTreSet(ByVal pTree As SsDbTre Ptr, ByRef key As String, ByVal value As Double, Opt ByVal DontReplace As Long)
    'add Key/Value to tree - Value replaced if Key exist unless DontReplace = True
    Local compare, temp As Long
    Local n As SsDbTreNode Ptr
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Sub
    If @pTree.root Then
        n = @pTree.root
        While 1
            Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
            If compare > 0 Then
                If @n.R Then
                    n = @n.R
                Else
                    @n.R = SsDbTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@R.P = n
                    SsSet @n.@R.K, key
                    @n.@R.V = value
                    SsDbTreBalanceBranch pTree, n
                    Exit Loop
                End If
            ElseIf compare < 0 Then
                If @n.L Then
                    n = @n.L
                Else
                    @n.L = SsDbTreAllocNode(pTree) : If Err Then Exit Sub
                    @n.@L.P = n
                    SsSet @n.@L.K, key
                    @n.@L.V = value
                    SsDbTreBalanceBranch pTree, n
                    Exit Loop
                End If
            Else
                If IsFalse DontReplace Then @n.V = value
                Exit Loop
            End If
        Wend
    Else
        @pTree.root = SsDbTreAllocNode(pTree) : If Err Then Exit Sub
        SsSet @pTree.@root.K, key
        @pTree.@root.V = value
        @pTree.count = 1
    End If
    temp = SsFinal(temp)
End Sub

Function SsDbTreGet(ByVal pTree As SsDbTre Ptr, ByRef key As String) As Double
    'get Key's associated Value
    Local compare, temp As Long
    Local n As SsDbTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = @n.V
            Exit Loop
        End If
    Wend
    temp = SsFinal(temp)
End Function

Function SsDbTreGot(ByVal pTree As SsDbTre Ptr, ByRef key As String) As Long
    'True/False if Key exist
    Local compare, temp As Long
    Local n As SsDbTreNode Ptr
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    temp = SsSetNew(key) : If Err Then Exit Function
    n = @pTree.root
    While n
        Call Dword @pTree.compareCB Using SsCompareCB(temp, @n.K, @pTree.collation) To compare
        If compare < 0 Then
            n = @n.L
        ElseIf compare > 0 Then
            n = @n.R
        Else
            Function = n
            Exit Loop
        End If
    Wend
    temp = SsFinal(temp)
End Function

Sub SsDbTreDel(ByVal pTree As SsDbTre Ptr, ByRef key As String)
    'remove Key and associated Value
    Local pNode As SsDbTreNode Ptr
    ExitS(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    pNode = SsDbTreGot(pTree, key)
    If pNode Then
        SsDbTreRemoveNode(pTree, pNode)
    End If
End Sub

Function SsDbTreFirst(ByVal pTree As SsDbTre Ptr) As Long
    'get handle to first node in tree
    Local n As SsDbTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.L
            n = @n.L
        Wend
    End If
    Function = n
End Function

Function SsDbTreLast(ByVal pTree As SsDbTre Ptr) As Long
    'get handle to last node in tree
    Local n As SsDbTreNode Ptr
    ExitF(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    n = @pTree.root
    If n Then
        While @n.R
            n = @n.R
        Wend
    End If
    Function = n
End Function

Function SsDbTreNext(ByVal pNode As SsDbTreNode Ptr) As Long
    'get handle to next node in tree
    Local minR As SsDbTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>SsDbTreNodeTag, LibErrH)
        minR = SsDbTreMinRight(pNode)
        If pNode <> minR Then
            Function = minR
        Else
            Function = SsDbTreParentGreater(pNode)
        End If
    End If
End Function

Function SsDbTrePrev(ByVal pNode As SsDbTreNode Ptr) As Long
    'get handle to previous node in tree
    Local maxL As SsDbTreNode Ptr
    If pNode Then
        ExitF(@pNode.tag<>SsDbTreNodeTag, LibErrH)
        maxL = SsDbTreMaxLeft(pNode)
        If pNode <> maxL Then
            Function = maxL
        Else
            Function = SsDbTreParentLesser(pNode)
        End If
    End If
End Function

Function SsDbTreGetKey(ByVal pNode As SsDbTreNode Ptr) As String
    'get node's Key
    ExitF(pNode=0 Or @pNode.tag<>SsDbTreNodeTag, LibErrH)
    Function = SsGet(@pNode.K)
End Function

Function SsDbTreGetVal(ByVal pNode As SsDbTreNode Ptr) As Double
    'get node's Value
    ExitF(pNode=0 Or @pNode.tag<>SsDbTreNodeTag, LibErrH)
    Function = @pNode.V
End Function

Sub SsDbTreSetVal(ByVal pNode As SsDbTreNode Ptr, ByVal value As Double)
    'get node's Value
    ExitS(pNode=0 Or @pNode.tag<>SsDbTreNodeTag, LibErrH)
    @pNode.V = value
End Sub

'++
    '----------------------------------------------------------------------------------------
    '   Clone Container
    '----------------------------------------------------------------------------------------
'--

Function SsDbTreClone(ByVal pTree As SsDbTre Ptr) As Long
    'create duplicate container
    Local h, clone As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    clone = SsDbTreNew() : If Err Then Exit Function
    h = SsDbTreFirst(pTree)
    While h
        SsDbTreSet clone, SsDbTreGetKey(h), SsDbTreGetVal(h)
        h = SsDbTreNext(h)
    Wend
    Function = clone
End Function

'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From String
    '----------------------------------------------------------------------------------------
'--

Function SsDbTreStore(ByVal pTree As SsDbTre Ptr) As String
    'store container to string
    Local h, keys, vals, stor As Long
    Err = 0
    ExitF(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    keys = SsLstNew() : If Err Then Exit Function
    vals = DbLstNew() : If Err Then Exit Function
    stor = SsLstNew() : If Err Then Exit Function
    If @pTree.count Then
        h = SsDbTreFirst(pTree)
        While h
            SsLstAdd keys, SsDbTreGetKey(h)
            DbLstAdd vals, SsDbTreGetVal(h)
            h = SsDbTreNext(h)
        Wend
        SsLstAdd stor, SsLstStore(keys)
        SsLstAdd stor, DbLstStore(vals)
        Function = SsLstStore(stor)
    End If
    keys = SsLstFinal(keys)
    vals = DbLstFinal(vals)
    stor = SsLstFinal(stor)
End Function

Sub SsDbTreRestore(ByVal pTree As SsDbTre Ptr, ByVal s As String)
    'restore container from string
    Local keys, vals, stor As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    SsDbTreClear pTree
    keys = SsLstNew() : If Err Then Exit Sub
    vals = DbLstNew() : If Err Then Exit Sub
    stor = SsLstNew() : If Err Then Exit Sub
    If Len(s) Then
        SsLstRestore stor, s : If Err Then Exit Sub
        ExitS(SsLstCount(stor)<>2, LibErrU)
        SsLstRestore keys, SsLstPopFirst(stor)
        DbLstRestore vals, SsLstPopFirst(stor)
        ExitS(SsLstCount(keys)<>DbLstCount(vals), LibErrU)
        While SsLstCount(keys)
            SsDbTreSet pTree, SsLstPopFirst(keys), DbLstPopFirst(vals)
        Wend
    End If
    keys = SsLstFinal(keys)
    vals = DbLstFinal(vals)
    stor = SsLstFinal(stor)
End Sub


'++
    '----------------------------------------------------------------------------------------
    '   Store/Restore Container To/From File
    '----------------------------------------------------------------------------------------
'--

Sub SsDbTreFileStore(ByVal pTree As SsDbTre Ptr, ByVal file As String)
    'store container to file
    Local s As String
    Local f As Long
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    s = SsDbTreStore(pTree) : If Err Then Exit Sub
    Try
        f = FreeFile
        Open file For Binary As f
        SetEof f
        Put$ f, s
        Close f
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

Sub SsDbTreFileRestore(ByVal pTree As SsDbTre Ptr, ByVal file As String)
    'restore container from file - Modifies Container Data
    Local f As Long
    Local s As String
    Err = 0
    ExitS(pTree=0 Or @pTree.tag<>SsDbTreTag, LibErrH)
    Try
        f = FreeFile
        Open file For Binary As f
        Get$ f, Lof(f), s
        SsDbTreRestore pTree, s
    Catch
        ExitLogErr(LibErrF)
    Finally
        If f Then Close f
    End Try
End Sub

    '----------------------------------------------------------------------------------------
    '   PRIVATE
    '----------------------------------------------------------------------------------------

Sub SsDbTreRemoveNode(ByVal p As SsDbTre Ptr, ByVal n As SsDbTreNode Ptr) Private
    ExitS(n=0, LibErrP)
    Local nP, swapN As SsDbTreNode Ptr
    While @n.L Or @n.R
        swapN = IIf&(@n.HL >= @n.HR, SsDbTreMaxLeft(n), SsDbTreMinRight(n))
        If @p.root = n Then @p.root = swapN
        Swap @n.K, @swapN.K
        Swap @n.V, @swapN.V
        n = swapN
    Wend
    If n = @p.root Then
        SsDbTreClear p
    Else
        nP = @n.P
        ExitS(nP=0, LibErrP)
        If @nP.L = n Then @nP.L = 0 Else @nP.R = 0
        n = SsDbTreFreeNode(p, n)
        SsDbTreBalanceBranch p, nP
    End If
End Sub

Function SsDbTreAllocNode(ByVal p As SsDbTre Ptr) Private As Long
    Local n As SsDbTreNode Ptr
    n = MemAlloc(SizeOf(SsDbTreNode))
    ExitF(n=0, LIbErrM)
    @n.HL = 1
    @n.HR = 1
    @n.tag = SsDbTreNodeTag
    @n.K = SsNew() : If Err Then Exit Function
    Incr @p.count
    Function = n
End Function

Function SsDbTreFreeNode(ByVal p As SsDbTre Ptr, ByVal n As SsDbTreNode Ptr) As Long
    If n Then
        @n.K = SsFinal(@n.K)
        MemFree(n)
        ExitF(@p.count=0, LibErrU)
        Decr @p.count
    End If
End Function

Sub SsDbTreBalanceBranch(ByVal p As SsDbTre Ptr, ByVal n As SsDbTreNode Ptr) Private
    While n
        @n.HL = IIf&(@n.L, Max&(@n.@L.HL, @n.@L.HR) + 1, 1)
        @n.HR = IIf&(@n.R, Max&(@n.@R.HL, @n.@R.HR) + 1, 1)
        If @n.HL > @n.HR + 1 Then
            n = SsDbTreRotateRight(p, n)
        ElseIf @n.HR > @n.HL + 1 Then
            n = SsDbTreRotateLeft(p, n)
        Else
            n = @n.P
        End If
    Wend
End Sub

Function SsDbTreMaxLeft(ByVal n As SsDbTreNode Ptr) Private As Long
    If n Then
        If @n.L Then
            n = @n.L
            While @n.R
                n = @n.R
            Wend
        End If
    End If
    Function = n
End Function

Function SsDbTreMinRight(ByVal n As SsDbTreNode Ptr) Private As Long
    If n Then
        If @n.R Then
            n = @n.R
            While @n.L
                n = @n.L
            Wend
        End If
    End If
    Function = n
End Function

Function SsDbTreParentGreater(ByVal n As SsDbTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.L = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function SsDbTreParentLesser(ByVal n As SsDbTreNode Ptr) Private As Long
    If n Then
        While @n.P
            If @n.@P.R = n Then
                Function = @n.P
                Exit Function
            Else
                n = @n.P
            End If
        Wend
    End If
End Function

Function SsDbTreRotateLeft(ByVal p As SsDbTre Ptr, ByVal n As SsDbTreNode Ptr) Private As Long
    Local nR, nRL As SsDbTreNode Ptr
    nR = @n.R
    If @nR.HL > @nR.HR Then
        nRL = @nR.L
        @n.R = nRL : @nRL.P = n
        @nR.L = @nRL.R : If @nR.L Then @nR.@L.P = nR
        @nRL.R = nR : @nR.P = nRL
        @nR.HL = IIf&(@nR.L, Max&(@nR.@L.HL, @nR.@L.HR) + 1, 1)
        @nR.HR = IIf&(@nR.R, Max&(@nR.@R.HL, @nR.@R.HR) + 1, 1)
        nR = @n.R
    End If
    If @p.root = n Then @p.root = @n.R
    @n.R = @nR.L : If @n.R Then @n.@R.P = n
    @nR.P = @n.P : @n.P = nR : @nR.L = n
    If @nR.P Then
        If @nR.@P.L = n Then @nR.@P.L = nR Else @nR.@P.R = nR
    End If
    Function = n
End Function

Function SsDbTreRotateRight(ByVal p As SsDbTre Ptr, ByVal n As SsDbTreNode Ptr) Private As Long
    Local nL, nLR As SsDbTreNode Ptr
    nL = @n.L
    If @nL.HR > @nL.HL Then
        nLR = @nL.R
        @n.L = nLR : @nLR.P = n
        @nL.R = @nLR.L : If @nL.R Then @nL.@R.P = nL
        @nLR.L = nL : @nL.P = nLR
        @nL.HL = IIf&(@nL.L, Max&(@nL.@L.HL, @nL.@L.HR) + 1, 1)
        @nL.HR = IIf&(@nL.R, Max&(@nL.@R.HL, @nL.@R.HR) + 1, 1)
        nL = @n.L
    End If
    If @p.root = n Then @p.root = @n.L
    @n.L = @nL.R : If @n.L Then @n.@L.P = n
    @nL.P = @n.P : @n.P = nL : @nL.R = n
    If @nL.P Then
        If @nL.@P.L = n Then @nL.@P.L = nL Else @nL.@P.R = nL
    End If
    Function = n
End Function
